package easik.sketch.constraint;


import java.util.ArrayList;
import java.util.LinkedList;

import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;
import easik.sketch.vertex.EntityNode;

/**
 * This class represents the sum colimit constraint. It contains a list of
 * paths sharing a target. 
 *
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-07-13 Vera Ranieri
 */
public class SumConstraint extends Constraint {
	

	/**
	 * Default Constructor takes an ArrayList of SketchPaths
	 * 
	 * @param inPaths An ArrayList of SketchPaths
	 */
	public SumConstraint(ArrayList<SketchPath> inPaths) {
		_paths = inPaths;
		_edges = new ArrayList<SketchEdge>();
		//Fill in _edges array
		for(SketchPath p : inPaths){
			LinkedList<SketchEdge> curPathEdges = p.getEdges();
			for(SketchEdge e: curPathEdges)
			{
				if(!_edges.contains(e))
					_edges.add(e);
			}
		}
		_isVisible = true;
		_constraintLabel = "+";
	}
	
	/**
	 * Constructor for specifying position on graph
	 * 
	 * @param inPaths An ArrayList of SketchPaths
	 * @param x X coordinate of visual on graph
	 * @param y Y coordinate of visual on graph
	 * @param isVisible If the constraint is visible in the graph or not
	 */
	public SumConstraint(ArrayList<SketchPath> inPaths, int x, int y, boolean isVisible) {
		super("+", x, y,isVisible);
		_paths = inPaths;
		_edges = new ArrayList<SketchEdge>();
			   
		//Fill in _edges array
		for(SketchPath p : inPaths){
			LinkedList<SketchEdge> curPathEdges = p.getEdges();
			for(SketchEdge e: curPathEdges)
			{
				if(!_edges.contains(e))
				_edges.add(e);
			}
		}
		_constraintLabel = "+";
	}
	
	/**
	 * Sets the path array, updates edge list, and updates display
	 * 
	 * @param inPaths The new array of paths
	 */
	public void setPaths(ArrayList<SketchPath> inPaths){
		_paths = inPaths;
		 
		addEdges();
		setVisible(!isVisible());
		setVisible(!isVisible());
	}
	
	/**
	 * Static method to determine whether a set of paths could theoretically 
	 * designate a sum constraint.
	 * @param paths The paths to check for a sum constraint formation.
	 * @return True if forms a sum constraint, false otherwise.
	 * @since 2006-05-29 Vera Ranieri
	 */
	public static boolean isSumConstraint(ArrayList paths){
//		We must have a bunch of injective arrows aimed at the same spot
		EntityNode sharedTarget = null;
		int edgesInvolved = 0;
		
		SketchPath currentPath;
		SketchEdge edge;

		int size = paths.size();
		// Search through the selection...
		for (int i = 0; i < size; i++) {
				currentPath = (SketchPath) paths.get(i);

				// First target we come across becomes our 'sharedTarget'
				if (sharedTarget == null) {
					sharedTarget = (EntityNode) currentPath.getCoDomain();
				} 
				else if (sharedTarget != currentPath.getCoDomain()) {
					// If we encounter a source which is not the sharedSource, fail			
					return false;
				}
				
				//Make sure the first edge in the path is injective
				if(currentPath.getEdges().size() != 0){
					edge = (SketchEdge)currentPath.getEdges().get(0);
					if(!edge.getInjective() )
						return false;
					
					for(int j = i+1; j< size; j++){
							if(SketchPath.pathsAreEqual(currentPath, (SketchPath)paths.get(j)))
								return false;
					}
				}
				// Otherwise, we're fine, add the edge
				edgesInvolved++;
		}

		if (edgesInvolved < 2) {
			return false;
		}
		
		return true;
	}
}

